<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\AdminSettingsModel;
use App\Models\LicenseKeyModel;
use App\Models\AuditLogModel;

class Settings extends BaseController
{
    protected $settingsModel;
    protected $licenseModel;
    protected $auditLog;

    public function __construct()
    {
        $this->settingsModel = new AdminSettingsModel();
        $this->licenseModel = new LicenseKeyModel();
        $this->auditLog = new AuditLogModel();
    }

    public function index()
    {
        $adminId = session()->get('admin_id');
        $settings = $this->settingsModel->getOrCreate($adminId);
        
        $licenseInfo = null;
        if ($settings['license_key']) {
            $licenseInfo = $this->licenseModel->where('code', $settings['license_key'])->first();
        }
        
        $systemUsage = [];
        if ($licenseInfo) {
            $systemUsage = $this->licenseModel->getSystemUsage($licenseInfo['id']);
        }
        
        return view('admin/settings/index', [
            'settings' => $settings,
            'licenseInfo' => $licenseInfo,
            'systemUsage' => $systemUsage
        ]);
    }

    public function updateLicenseKey()
    {
        $adminId = session()->get('admin_id');
        $licenseKey = $this->request->getPost('license_key');
        
        $license = $this->licenseModel->isValidLicense($licenseKey);
        
        if (!$license) {
            return redirect()->back()->with('error', 'Invalid or expired license key');
        }
        
        $oldSettings = $this->settingsModel->where('admin_id', $adminId)->first();
        
        $this->settingsModel->where('admin_id', $adminId)->set([
            'license_key' => $licenseKey,
            'updated_at' => date('Y-m-d H:i:s')
        ])->update();
        
        $db = \Config\Database::connect();
        $db->table('admins')->where('id', $adminId)->update(['license_id' => $license['id']]);
        
        $this->auditLog->log('admin', $adminId, 'Updated License Key', 'admin_settings', $oldSettings['id'], $oldSettings, ['license_key' => $licenseKey]);
        
        return redirect()->to('/admin/settings')->with('success', 'License key updated successfully');
    }

    public function updateSettings()
    {
        $adminId = session()->get('admin_id');
        $oldSettings = $this->settingsModel->where('admin_id', $adminId)->first();
        
        $data = [
            'allow_login' => $this->request->getPost('allow_login') === 'yes',
            'camera_detection' => $this->request->getPost('camera_detection') === 'yes',
            'person_detection' => $this->request->getPost('person_detection') === 'yes',
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->settingsModel->where('admin_id', $adminId)->set($data)->update();
        
        $this->auditLog->log('admin', $adminId, 'Updated Settings', 'admin_settings', $oldSettings['id'], $oldSettings, $data);
        
        return redirect()->to('/admin/settings')->with('success', 'Settings updated successfully');
    }

    public function validateLicenseKey()
    {
        $licenseKey = $this->request->getGet('code');
        $license = $this->licenseModel->where('code', $licenseKey)->first();
        
        if (!$license) {
            return $this->response->setJSON(['valid' => false, 'message' => 'License key not found']);
        }
        
        $isValid = $this->licenseModel->isValidLicense($licenseKey);
        
        return $this->response->setJSON([
            'valid' => $isValid !== false,
            'license' => $license
        ]);
    }
}
